'use strict';

/* --------------------------------------------------------------
 modal.js 2016-07-07
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2016 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

jse.libs.theme.modal = jse.libs.theme.modal || {};

/**
 * ## Honeygrid Modal Dialogs Library
 *
 * Library-function to open default modal layer.  This function depends on jQuery & jQuery UI.
 *
 * @module Honeygrid/Libs/modal
 * @exports jse.libs.theme.modal
 * @ignore
 */
(function (exports) {
    'use strict';

    var $body = $('body'),
        tplStore = [],
        extension = null,

    // Object for default buttons
    buttons = {
        yes: {
            name: jse.core.lang.translate('yes', 'buttons'),
            type: 'success',
            class: 'btn-success'
        },
        no: {
            name: jse.core.lang.translate('no', 'buttons'),
            type: 'fail',
            class: 'btn-default'
        },
        abort: {
            name: jse.core.lang.translate('abort', 'buttons'),
            type: 'fail',
            class: 'btn-default'
        },
        ok: {
            name: jse.core.lang.translate('ok', 'buttons'),
            type: 'success',
            class: 'btn-success'
        },
        close: {
            name: jse.core.lang.translate('close', 'buttons'),
            type: 'fail',
            class: 'btn-default'
        }
    };

    /**
     *    Function to get all form data stored inside
     *    the layer
     *
     *    @param        {object}    $self        jQuery selection of the layer
     *    @return    {json}                    Returns a JSON with all form data
     */
    var _getFormData = function _getFormData($self, checkform) {
        var $forms = $self.filter('form').add($self.find('form')),
            formdata = {},
            valid = true,
            promises = [];

        if ($forms.length) {
            $forms.each(function () {
                var $form = $(this);

                if (checkform) {
                    var localDeferred = $.Deferred();
                    promises.push(localDeferred);
                    $form.trigger('validator.validate', { deferred: localDeferred });
                }

                formdata[$form.attr('name') || $form.attr('id') || 'form_' + new Date().getTime() * Math.random()] = jse.libs.form.getData($form);
            });
        }

        return $.when.apply(undefined, promises).then(function () {
            return formdata;
        }, function () {
            return formdata;
        }).promise();
    };

    /**
     *    Function to transform the custom buttons object (which is
     *    incompatible with jQuery UI) to a jQuery UI compatible format
     *
     *    @param        {object}    dataset        Custom buttons object for the dialog
     *    @param        {promise}    deferred    deferred-object to resolve / reject on close
     *    @return    {array}                    Returns a jQuery UI dialog compatible buttons array
     */
    var _genButtons = function _genButtons(options, extensionDeferred) {

        // Check if buttons are available
        if (options.buttons) {

            var rejectHandler = extension.getRejectHandler,
                resolveHandler = extension.getResolveHandler;

            $.each(options.buttons, function (k, v) {

                // Setup click handler
                options.buttons[k].event = function () {
                    var $self = $(this);

                    // If a callback is given, execute it with
                    // the current scope
                    if (typeof v.callback === 'function') {
                        if (!v.callback.apply($self, [])) {
                            return false;
                        }
                    }

                    // Add the default behaviour
                    // for the close  functionality
                    // On fail, reject the deferred
                    // object, else resolve it
                    switch (v.type) {
                        case 'fail':
                            rejectHandler($self, extensionDeferred, _getFormData);
                            break;
                        case 'success':
                            resolveHandler($self, extensionDeferred, _getFormData);
                            break;
                        case 'link':
                            location.href = v.value;
                            break;
                        default:
                            break;
                    }
                };
            });
        }
    };

    var _finalizeLayer = function _finalizeLayer($container, options) {
        // Prevent submit on enter in inner forms
        var $forms = $container.find('form');
        if ($forms.length) {
            $forms.on('submit', function (e) {
                e.preventDefault();
            });
        }

        if (window.gambio && window.gambio.widgets && window.gambio.widgets.init) {
            window.gambio.widgets.init($container);
        }
    };

    var _setLayer = function _setLayer(name) {
        if (jse.libs.theme.modal[name]) {
            extension = jse.libs.theme.modal[name];
        } else {
            jse.core.debug.error('[MODAL] Can\'t set modal: "' + name + '". Extension doesn\'t exist');
        }
    };

    var _transferOptions = function _transferOptions(options) {
        var mapper = extension.getMapper(),
            result = {};

        $.each(options, function (k, v) {

            if (mapper[k] === false) {
                return true;
            } else if (mapper[k] === undefined) {
                result[k] = v;
            } else if (typeof mapper[k] === 'function') {
                var mapperResult = mapper[k](k, v);
                result[mapperResult[0]] = mapperResult[1];
            } else {
                result[mapper[k]] = v;
            }
        });

        return result;
    };

    var _getTemplate = function _getTemplate(options, iframe) {

        var $selection = [],
            deferred = $.Deferred();

        if (options.noTemplate) {
            deferred.resolve('');
        } else if (iframe) {
            deferred.resolve('<iframe width="100%" height="100%" frameborder="0" src="' + options.template + '" />');
        } else {
            if (options.storeTemplate && tplStore[options.template]) {
                deferred.resolve(tplStore[options.template]);
            } else {

                try {
                    $selection = $(options.template);
                } catch (err) {}

                if ($selection.length) {
                    deferred.resolve($selection.html());
                } else {
                    jse.libs.xhr.ajax({ url: options.template, dataType: 'html' }).done(function (result) {
                        if (options.sectionSelector) {
                            result = $(result).find(options.sectionSelector).html();
                        }

                        if (options.storeTemplate) {
                            tplStore[options.template] = result;
                        }
                        deferred.resolve(result);
                    }).fail(function () {
                        deferred.reject();
                    });
                }
            }
        }

        return deferred;
    };

    var _createLayer = function _createLayer(options, title, className, defbuttons, template) {
        // Setup defaults & deferred objects
        var deferred = $.Deferred(),
            promise = deferred.promise(),
            iframe = template === 'iframe',
            defaults = {
            title: title,
            dialogClass: className,
            modal: true,
            buttons: defbuttons || [],
            closeOnEscape: true,
            template: template || null,
            storeTemplate: false,
            closeX: true,
            closeOnOuter: true
        },
            instance = null,
            $forms = null,
            extensionDeferred = $.Deferred();

        // Merge custom settings with default settings
        options = options || {};
        options = $.extend({}, defaults, options);

        var tplRequest = _getTemplate(options, iframe).done(function (result) {

            extensionDeferred.done(function (result) {
                deferred.resolve(result);
            }).fail(function (result) {
                deferred.reject(result);
            });

            // Generate template
            options.template = $(Mustache.render(result, options));
            jse.libs.theme.helpers.setupWidgetAttr(options.template);
            options.template = $('<div>').append(options.template.clone()).html();

            // Generate default button object
            _genButtons(options, extensionDeferred);

            // Transfer options object to extension option object
            var originalOptions = $.extend({}, options);
            options = _transferOptions(options);

            // Call extension
            extension.openLayer(options, extensionDeferred, _getFormData, originalOptions);

            // Passthrough of the close method of the layer
            // to the layer caller
            promise.close = function (success) {
                extensionDeferred.close(success);
            };
        }).fail(function () {
            deferred.reject({ error: 'Template not found' });
        });

        // Temporary close handler if the upper
        // deferred isn't finished now. It will be
        // overwritten after the layer opens
        if (!promise.close) {
            promise.close = function () {
                tplRequest.reject('Closed after opening');
            };
        }

        return promise;
    };

    /**
     *    Shortcut function for an alert-layer
     *    @param        {object}    options Options that are passed to the modal layer
     *    @return    {promise}            Returns a promise
     */
    var _alert = function _alert(options) {
        return _createLayer(options, jse.core.lang.translate('hint', 'labels'), '', [buttons.close], '#modal_alert');
    };

    /**
     *    Shortcut function for an confirm-layer
     *    @param        {object}    options Options that are passed to the modal layer
     *    @return    {promise}            Returns a promise
     */
    var _confirm = function _confirm(options) {
        return _createLayer(options, jse.core.lang.translate('confirm', 'labels'), 'confirm_dialog', [buttons.yes, buttons.no], '#modal_alert');
    };

    /**
     *    Shortcut function for a prompt-layer
     *    @param        {object}    options Options that are passed to the modal layer
     *    @return    {promise}            Returns a promise
     */
    var _prompt = function _prompt(options) {
        return _createLayer(options, jse.core.lang.translate('prompt', 'labels'), 'prompt_dialog', [buttons.ok, buttons.abort], '#modal_prompt');
    };

    /**
     *    Shortcut function for an success-layer
     *    @param        {object}    options Options that are passed to the modal layer
     *    @return    {promise}            Returns a promise
     */
    var _success = function _success(options) {
        return _createLayer(options, jse.core.lang.translate('success', 'labels'), 'success_dialog', [], '#modal_alert');
    };

    /**
     *    Shortcut function for an error-layer
     *    @param        {object}    options Options that are passed to the modal layer
     *    @return    {promise}            Returns a promise
     */
    var _error = function _error(options) {
        return _createLayer(options, jse.core.lang.translate('errors', 'labels'), 'error_dialog', [], '#modal_alert');
    };

    /**
     *    Shortcut function for a warning-layer
     *    @param        {object}    options Options that are passed to the modal layer
     *    @return    {promise}            Returns a promise
     */
    var _warn = function _warn(options) {
        return _createLayer(options, jse.core.lang.translate('warning', 'labels'), 'warn_dialog', [], '#modal_alert');
    };

    /**
     *    Shortcut function for an info-layer
     *    @param        {object}    options Options that are passed to the modal layer
     *    @return    {promise}            Returns a promise
     */
    var _info = function _info(options) {
        return _createLayer(options, jse.core.lang.translate('info', 'labels'), 'info_dialog', [], '#modal_alert');
    };

    /**
     *    Shortcut function for an iframe-layer
     *    @param        {object}    options Options that are passed to the modal layer
     *    @return    {promise}            Returns a promise
     */
    var _iframe = function _iframe(options) {
        if (options.convertModal) {
            jse.libs.theme.modal[options.convertModal](options, jse.core.lang.translate('info', 'labels'), options.convertModal + '_dialog', [], '#modal_alert');
            return;
        }

        return _createLayer(options, jse.core.lang.translate('info', 'labels'), 'iframe_layer', [], 'iframe');
    };

    // ########## VARIABLE EXPORT ##########

    exports.error = _error;
    exports.warn = _warn;
    exports.info = _info;
    exports.success = _success;
    exports.alert = _alert;
    exports.prompt = _prompt;
    exports.confirm = _confirm;
    exports.iframe = _iframe;
    exports.custom = _createLayer;
    exports.setLayer = _setLayer;
    exports.finalizeLayer = _finalizeLayer;

    // Set default layer.
    var currentTimestamp = Date.now,
        lifetime = 10000; // 10 sec

    extension = jse.core.registry.get('mainModalLayer');

    var intv = setInterval(function () {
        if (jse.libs.theme.modal[extension] !== undefined) {
            _setLayer(extension);
            clearInterval(intv);
        }

        if (Date.now - currentTimestamp > lifetime) {
            throw new Error('Modal extension was not loaded: ' + extension);
        }
    }, 300);
})(jse.libs.theme.modal);

jse.libs.template = jse.libs.template || {};
jse.libs.template.modal = jse.libs.theme.modal;
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
